defmodule Regles do
  @moduledoc """
  Modèle de comportement des règles du jeu.

  Tous les jeux doivent implémenter ce comportement.
  """

  @typedoc "Un coup utilisable en jeu"
  @type coup :: atom() | tuple()

  @doc """
  Retourne le titre du jeu sous la forme d'une chaîne de caractères.
  """
  @callback titre() :: String.t()

  @doc """
  Retourne une version condensée de notre jeu de cartes.
  """
  @callback condensée(struct(), integer()) :: map()

  @doc """
  Crée une nouvelle structure.

  Cette fonction retourne la structure propre au jeu de règles
  (probablement définie dans le module spécifique).
  """
  @callback new() :: struct()

  @doc """
  Retourne `true` si on peut ajouter un nouveau joueur au jeu, `false` sinon.
  """
  @callback ajouter_joueur?(struct()) :: boolean()

  @doc """
  Ajoute un joueur.

  On doit préciser en paramètre le jeu lui-même et le nom du joueur
  à ajouter. `ajouter_joueur?` doit avoir été appelée auparavant.
  Un tuple est retourné contenant le jeu modifié et l'identifiant
  du joueur ajouté (un entier).
  """
  @callback ajouter_joueur(struct(), String.t()) :: {struct(), integer()}

  @doc """
  Est-ce que ce coup est permis ?

  Retourne soit `true` soit `false`.
  On doit préciser en paramètres le jeu en lui-même, le joueur sous forme
  d'entier et le coup à jouer : ce dernier peut être un atome (comme
  `:piocher`) ou un tuple (comme `{:jouer, %Carte{}}`).
  """
  @callback jouer?(struct(), integer(), coup()) :: boolean()

  @doc """
  Retourne les coups pour le joueur sélectionné.

  Cette fonction prend en paramètre le jeu et l'identifiant du joueur.
  Elle retourne une liste de coup, chaque coup étant un tuple représentant
  la référence du coup (à envoyer à `joueur`), le nom du coup
  (pour l'affichage) et un booléen indiquant si ce coup est valide ou non
  à ce moment.
  """
  @callback coups(struct(), integer()) :: [{atom() | tuple(), String.t(), boolean()}]

  @doc """
  Fait jouer un joueur.

  On doit spécifier en paramètres le jeu lui-même, le joueur sous la forme
  d'un entier et le coup à réaliser (un atome ou un tuple
  dont le premier élément est un atome).
  La fonction `jouer?` doit avoir été appelée auparavant.
  Le jeu modifié est retourné.
  """
  @callback jouer(struct(), integer(), coup()) :: struct()
end
